/* 
 * GeneralPropertiesReader.java        0.1.4 12/02/16
 * 
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.config;

import de.esukom.decoit.ifmapclient.logging.Logging;
import de.esukom.decoit.ifmapclient.main.IfMapClient;
import de.esukom.decoit.ifmapclient.messaging.SOAPMessageSender;
import de.esukom.decoit.ifmapclient.util.Toolbox;

import java.io.FileInputStream;
import java.util.Properties;
import java.util.logging.Logger;

/**
 * Reads properties from configuration file
 * 
 * @version 0.1.4
 * @author Tobias, FHH/TRUST
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class GeneralPropertiesReader {

    public static final Logger LOGGER = Logging.getTheLogger();

    // default values for empty or "wrong" properties
    private static final int APPLICATION_RETRYCOUNT_DEFAULT = 3;
    private static final int APPLICATION_POLLINGINTERVAL_DEFAULT = 10;
    private static final boolean APPLICATION_RENEWSESSION_DEFAULT = false;
    private static final boolean APPLICATION_ARCPOLLING_DEFAULT = false;
    private static final int APPLICATION_RENEWSESSIONINTERVAL_DEFAULT = 30;
    private static final boolean MAPSERVER_BASICAUTHENABLED_DEFAULT = true;
    private static final boolean APPLICATION_MESSAGING_SENDOLD = false;
    //private static final String APPLICATION_SNORT_MODE = "default";
    //private static final String APPLICATION_IPTABLES_MODE = "default";

    private static Properties sProps;
    private static boolean sFlag = false;
    private static String sPath = "config/config.properties";

    /**
     * Sets the path to the properties file.
     */
    public static void setPath(String path) {
        GeneralPropertiesReader.sPath = path;
    }

    /**
     * Loads the properties to memory
     * 
     * @param path
     *            path of the property file
     */
    public static void loadProperties(String path) {
        sProps = new Properties();
        FileInputStream in;
        try {
            in = new FileInputStream(path);
            sProps.load(in);
            in.close();
            initConfig(sProps);
            sFlag = true;
        } catch (Exception e) {
            e.printStackTrace();
            IfMapClient.exit("error while loading general properties from " + path + "...please check your configuration");
        }
    }

    /**
     * initialize configuration from property-file
     * 
     * @param conf
     *            properties-object
     */
    private static void initConfig(Properties conf) {
        if (!initConfigFiles(conf) || !initApplicationConfig(conf) || !initMapServerConfig(conf)) {
            IfMapClient.exit("error while initializing configuration");
        }
    }

    /**
     * get config-properties from configuration file and set them to GeneralConfig-Class.
     * 
     * @param conf
     *            properties-object
     * 
     * @return true if successful, false otherwise
     */
    private static boolean initConfigFiles(Properties conf) {
        // configure parameters via command line arguments
        if (IfMapClient.sInputConfigParams != null && buildConfigFileOptionsFromCommandLineParameters()) {
            return true;
        } else {
            // path to polling configuration
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = Toolbox.getStringProperty("application.pollingconfig.path", conf, true);
            if (GeneralConfig.APPLICATION_POLLINGCONFIG_PATH == null) {
                return false;
            }

            // path to mapping configuration
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = Toolbox.getStringProperty("application.mappingconfig.path", conf, true);
            if (GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH == null) {
                return false;
            }

            // class-name for polling class
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = Toolbox.getStringProperty("application.pollingconfig.classname", conf,
                    false);
            if (GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME == null) {
                return false;
            }

            // class-name for mapping class
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = Toolbox.getStringProperty("application.mappingconfig.classname", conf,
                    false);
            if (GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME == null) {
                return false;
            }
        }
        return true;
    }

    /**
     * use command line parameters for detecting polling and mappong properties
     * 
     * @return true, if config was detected and set, false otherweise
     */
    private static boolean buildConfigFileOptionsFromCommandLineParameters() {
        IfMapClient.LOGGER.info("Detected Command Line Configuration Parameter: ");
        for (int i = 0; i < IfMapClient.sInputConfigParams.length; i++) {
            IfMapClient.LOGGER.info("Entry[" + i + "]:" + IfMapClient.sInputConfigParams[i].toString());
        }

        boolean isFirstInputParamValid = false;

        // component-configuration
        if (IfMapClient.sInputConfigParams[0].equals("snort-ascii")) {
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = "config/snort/file_polling.properties";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = "config/snort/mapping.properties";
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.pollingthreads.SnortFilePollingThread";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.mappingfactory.SnortFileEventMappingFactory";
            IfMapClient.LOGGER.info("Configured snort-ascii");
            isFirstInputParamValid = true;
        } else if (IfMapClient.sInputConfigParams[0].equals("snort-barnyard")) {
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = "config/snort/file_polling.properties";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = "config/snort/mapping.properties";
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.pollingthreads.SnortBarnyardFilePollingThread";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.mappingfactory.SnortBarnyardFileEventMappingFactory";
            IfMapClient.LOGGER.info("Configured snort-barnyard");
            isFirstInputParamValid = true;
        } else if (IfMapClient.sInputConfigParams[0].equals("snort-sql")) {
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = "config/snort/db_polling.properties";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = "config/snort/mapping.properties";
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.pollingthreads.SnortSqlPollingThread";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.mappingfactory.SnortSqlEventMappingFactory";
            IfMapClient.LOGGER.info("Configured snort-sql");
            isFirstInputParamValid = true;
        } else if (IfMapClient.sInputConfigParams[0].equals("nagios")) {
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = "config/nagios/socket_polling.properties";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = "config/nagios/mapping.properties";
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.pollingthreads.NagiosSocketPollingThread";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.mappingfactory.NagiosEventMappingFactory";
            IfMapClient.LOGGER.info("Configured nagios");
            isFirstInputParamValid = true;
        } else if (IfMapClient.sInputConfigParams[0].equals("iptables")) {
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = "config/iptables/file_polling.properties";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = "config/iptables/mapping.properties";
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.pollingthreads.IPTablesULogFilePollingThread";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.mappingfactory.IPTablesEventMappingFactory";
            IfMapClient.LOGGER.info("Configured iptables");
            isFirstInputParamValid = true;
        } else if (IfMapClient.sInputConfigParams[0].equals("radius-file")) {
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = "config/radius/file_polling.properties";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = "config/radius/mapping.properties";
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.pollingthreads.RadiusFilePollingThread";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.mappingfactory.RadiusEventMappingFactory";
            IfMapClient.LOGGER.info("Configured radius-file");
            isFirstInputParamValid = true;
        } else if (IfMapClient.sInputConfigParams[0].equals("radius-sql")) {
            GeneralConfig.APPLICATION_POLLINGCONFIG_PATH = "config/radius/sql_polling.properties";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH = "config/radius/mapping.properties";
            GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.pollingthreads.RadiusSqlPollingThread";
            GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME = "de.esukom.decoit.ifmapclient.mappingfactory.RadiusSQLEventMappingFactory";
            IfMapClient.LOGGER.info("Configured radius-sql");
            isFirstInputParamValid = true;
        }

        // mode-configuration
        if (IfMapClient.sInputConfigParams.length > 1 && isFirstInputParamValid) {
            if (IfMapClient.sInputConfigParams[1].equals("esukom-mode")) {
                GeneralConfig.APPLICATION_IPTABLES_MODE = "esukom";
                GeneralConfig.APPLICATION_SNORT_MODE = "esukom";
                IfMapClient.LOGGER.info("Configured esukom-mode");
            } else if (IfMapClient.sInputConfigParams[1].equals("default-mode")) {
                GeneralConfig.APPLICATION_IPTABLES_MODE = "default";
                GeneralConfig.APPLICATION_SNORT_MODE = "default";
                IfMapClient.LOGGER.info("Configured default-mode");
            } else {
                GeneralConfig.APPLICATION_IPTABLES_MODE = "default";
                GeneralConfig.APPLICATION_SNORT_MODE = "default";
                IfMapClient.LOGGER.info("unknown second input parameter...configured default-mode");
            }
            return true;
        // no second paramter - use value from config-file
        } else if (IfMapClient.sInputConfigParams.length == 1 && isFirstInputParamValid) {
            GeneralConfig.APPLICATION_IPTABLES_MODE = "default";
            GeneralConfig.APPLICATION_SNORT_MODE = "default";
            IfMapClient.LOGGER.info("Configured default-mode");
            return true;
        }

        // wrong input-params
        IfMapClient.LOGGER.warning("Command Line Configuration Parameter not valid...using config-file values instead!");
        return false;
    }

    /**
     * get application-properties from configuration file and set them to GeneralConfig-Class.
     * 
     * @param conf
     *            properties-object
     * 
     * @return true if successful, false otherwise
     */
    private static boolean initApplicationConfig(Properties conf) {
        // application version string
        GeneralConfig.APPLICATION_VERSION_STRING = Toolbox.getStringProperty("application.version", conf, false);

        // connection retry-count
        GeneralConfig.APPLICATION_CONNECTION_RETRYCOUNT = Toolbox.getIntPropertyWithDefault("application.connection.retrycount",
                APPLICATION_RETRYCOUNT_DEFAULT, conf, true);

        // polling interval
        GeneralConfig.APPLICATION_POLLING_INTERVAL = Toolbox.getIntPropertyWithDefault("application.polling.interval",
                APPLICATION_POLLINGINTERVAL_DEFAULT, conf, true);

        // renew session enabled flag
        GeneralConfig.APPLICATION_RENEWSESSION_ENABLED = Toolbox.getBoolPropertyWithDefault("application.renewsession.enabled",
                APPLICATION_RENEWSESSION_DEFAULT, conf);

        // arc-polling enabled flag
        // if the polling and mapping components are initialized via command
        // line parameter and equals "iptables", this flag must be set to true!
        if (IfMapClient.sInputConfigParams != null && IfMapClient.sInputConfigParams.equals("iptables")) {
            GeneralConfig.APPLICATION_ARCPOLLING_ENABLED = true;
        } else {
            GeneralConfig.APPLICATION_ARCPOLLING_ENABLED = Toolbox.getBoolPropertyWithDefault("application.arcpolling.enabled",
                    APPLICATION_ARCPOLLING_DEFAULT, conf);
        }

        // skip/post "old" entries flag
        GeneralConfig.APPLICATION_MESSAGING_SENDOLD = Toolbox.getBoolPropertyWithDefault("application.messaging.sendold",
                APPLICATION_MESSAGING_SENDOLD, conf);

        // renew session interval
        GeneralConfig.APPLICATION_RENEWSESSION_INTERVALL = Toolbox.getIntPropertyWithDefault("application.renewsession.intervall",
                APPLICATION_RENEWSESSIONINTERVAL_DEFAULT, conf, true);

        // ip-address of the machine this client runs on
        GeneralConfig.APPLICATION_IPADDRESS = Toolbox.getStringPropertyWithDefault("application.ipaddress", conf, "unknown");

        // snort-publishing mode
        GeneralConfig.APPLICATION_SNORT_MODE = Toolbox.getStringPropertyWithDefault("application.snort.mode", conf, "default");

        // iptables-mode
        GeneralConfig.APPLICATION_IPTABLES_MODE = Toolbox.getStringPropertyWithDefault("application.iptables.mode", conf, "default");
        
        // iptables-startup-rules-script to be executed public static String APPLICATION_IPTABLES_STARTSCRIPT;
        GeneralConfig.APPLICATION_IPTABLES_STARTSCRIPT = Toolbox.getStringPropertyWithDefault("application.iptables.startscript", conf, "config/iptables/intialize_rules.sh");
        
        // iptables-sending of Datastream-Detected-Events
        GeneralConfig.APPLICATION_IPTABLES_SENDDATASTREAMDETECTEDEVENT = Toolbox.getBoolPropertyWithDefault("application.iptables.senddatastreamevents", false, conf);
        
        // gateway mode for ip-tables-component
        GeneralConfig.APPLICATION_IPTABLES_GATEWAYMODE = Toolbox.getBoolPropertyWithDefault("application.iptables.gateway", false, conf);

        // IF-MAP messaging type...can be null (no default)
        GeneralConfig.APPLICATION_MESSAGING_TYPE = Toolbox.getStringProperty("application.messaging.type", conf, false);
        if (GeneralConfig.APPLICATION_MESSAGING_TYPE == null
                || (!GeneralConfig.APPLICATION_MESSAGING_TYPE.equalsIgnoreCase("update") && !GeneralConfig.APPLICATION_MESSAGING_TYPE
                        .equalsIgnoreCase("notify"))) {
            return false;
        }
        // set publish type (now!)
        else {
            if (GeneralConfig.APPLICATION_MESSAGING_TYPE.equalsIgnoreCase("update")) {
                SOAPMessageSender.publishMode = SOAPMessageSender.PUBLISH_TYPE_UPDATE;
            } else {
                SOAPMessageSender.publishMode = SOAPMessageSender.PUBLISH_TYPE_NOTIFY;
            }
        }

        return true;
    }

    /**
     * get mapserver-properties from configuration file and set them to GeneralConfig-Class.
     * 
     * @param conf
     *            properties-object
     * 
     * @return true if successful, false otherwise
     */
    private static boolean initMapServerConfig(Properties conf) {
        // map-server url
        GeneralConfig.MAPSERVER_URL = Toolbox.getStringProperty("mapserver.url", conf, false);
        if (GeneralConfig.MAPSERVER_URL == null) {
            return false;
        }

        // if configuration was partly done by using command line arguments
        if (IfMapClient.sInputConfigParams != null && buildSSLConfigFromCommandLineParamaeter()) {
            // move along, nothing to see here...
        } else {
            // mapserver keystore path
            GeneralConfig.MAPSERVER_KEYSTORE_PATH = Toolbox.getStringProperty("mapserver.keystore.path", conf, true);
            if (GeneralConfig.MAPSERVER_KEYSTORE_PATH == null) {
                return false;
            }

            // mapserver keystore password
            GeneralConfig.MAPSERVER_KEYSTORE_PASSWORD = Toolbox.getStringProperty("mapserver.keystore.password", conf, false);
            if (GeneralConfig.MAPSERVER_KEYSTORE_PASSWORD == null) {
                return false;
            }

            // mapserver truststore path
            GeneralConfig.MAPSERVER_TRUSTSTORE_PATH = Toolbox.getStringProperty("mapserver.truststore.path", conf, true);
            if (GeneralConfig.MAPSERVER_TRUSTSTORE_PATH == null) {
                return false;
            }

            // mapserver truststore password
            GeneralConfig.MAPSERVER_TRUSTSTORE_PASSWORD = Toolbox.getStringProperty("mapserver.truststore.password", conf, false);
            if (GeneralConfig.MAPSERVER_TRUSTSTORE_PASSWORD == null) {
                return false;
            }
        }
        // basic authorization flag
        GeneralConfig.MAPSERVER_BASICAUTH_ENABLED = Toolbox.getBoolPropertyWithDefault("mapserver.basicauth.enabled",
                MAPSERVER_BASICAUTHENABLED_DEFAULT, conf);

        // basic authorization user
        GeneralConfig.MAPSERVER_BASICAUTH_USER = Toolbox.getStringProperty("mapserver.basicauth.user", conf, false);
        if (GeneralConfig.MAPSERVER_BASICAUTH_USER == null) {
            return false;
        }

        // basic authorization password
        GeneralConfig.MAPSERVER_BASICAUTH_PASSWORD = Toolbox.getStringProperty("mapserver.basicauth.password", conf, false);
        if (GeneralConfig.MAPSERVER_BASICAUTH_PASSWORD == null) {
            return false;
        }

        return true;
    }

    /**
     * use command line parameters for detecting polling and mappong properties
     * 
     * @return true, if config was detected and set, false otherweise
     */
    private static boolean buildSSLConfigFromCommandLineParamaeter() {
        IfMapClient.LOGGER.info("Detected Command Line Configuration Parameter: " + IfMapClient.sInputConfigParams);
        if (IfMapClient.sInputConfigParams[0].equals("snort-ascii") || IfMapClient.sInputConfigParams[0].equals("snort-barnyard")
                || IfMapClient.sInputConfigParams[0].equals("snort-sql")) {
            GeneralConfig.MAPSERVER_KEYSTORE_PATH = "keystore/snortmap.jks";
            GeneralConfig.MAPSERVER_KEYSTORE_PASSWORD = "snortmap";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PATH = "keystore/snortmap.jks";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PASSWORD = "snortmap";
            IfMapClient.LOGGER.info("using snortmap as key and trusstore with password snortmap");
            return true;
        } else if (IfMapClient.sInputConfigParams[0].equals("nagios")) {
            GeneralConfig.MAPSERVER_KEYSTORE_PATH = "keystore/nagiosmap.jks";
            GeneralConfig.MAPSERVER_KEYSTORE_PASSWORD = "nagiosmap";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PATH = "keystore/nagiosmap.jks";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PASSWORD = "nagiosmap";
            IfMapClient.LOGGER.info("using nagiosmap as key and trusstore with password nagiosmap");
            return true;
        } else if (IfMapClient.sInputConfigParams[0].equals("iptables")) {
            GeneralConfig.MAPSERVER_KEYSTORE_PATH = "keystore/iptablesmap.jks";
            GeneralConfig.MAPSERVER_KEYSTORE_PASSWORD = "iptablesmap";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PATH = "keystore/iptablesmap.jks";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PASSWORD = "iptablesmap";
            IfMapClient.LOGGER.info("Cusing iptables as key and trusstore with password iptablesmap");
            return true;
        } else if (IfMapClient.sInputConfigParams[0].equals("radius-sql")) {
            GeneralConfig.MAPSERVER_KEYSTORE_PATH = "keystore/freeradius.jks";
            GeneralConfig.MAPSERVER_KEYSTORE_PASSWORD = "freeradius";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PATH = "keystore/freeradius.jks";
            GeneralConfig.MAPSERVER_TRUSTSTORE_PASSWORD = "freeradius";
            IfMapClient.LOGGER.info("using radius as key and trusstore with password freeradius");
            return true;
        } else {
            IfMapClient.LOGGER.warning("Command Line Configuration Parameter not valid...using config-file values instead!");
            return false;
        }
    }

    /**
     * Gets a specific property
     * 
     * @param ident
     *            The property name
     * @return String the value of a property
     */

    public static String getProperty(String ident) {
        if (!sFlag) {
            loadProperties(GeneralPropertiesReader.sPath);
        }
        String value = sProps.getProperty(ident);
        return value;
    }

}
