/* 
 * SnortSqlPollingThread.java        0.1.4 12/02/16
 * 
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */
package de.esukom.decoit.ifmapclient.pollingthreads;

import de.esukom.decoit.ifmapclient.main.IfMapClient;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;

/**
 * Thread for polling the DB for new Snort-Events
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class SnortSqlPollingThread extends PollingThread {

    // properties from config file
    private String mSnortDatabaseName;
    private String mSnortDatabaseHost;
    private String mSnortDatabaseUser;
    private String mSnortDatabasePass;

    // database-parameters
    private Connection mConnect = null;
    private PreparedStatement mPrepStatement = null;

    // id to begin with when executing database-statement
    private int mLastCid = 0; // startup-value

    // SQL statement, used for building a prepared statement
    private final String mRawQuery = "SELECT event.sid,sensor.hostname,event.cid,event.timestamp,signature.sig_name,signature.sig_priority,sig_class.sig_class_name,inet_ntoa(iphdr.ip_src),tcphdr.tcp_sport,inet_ntoa(iphdr.ip_dst),tcphdr.tcp_dport,iphdr.ip_ver,reference_system.ref_system_name,reference.ref_tag FROM event INNER JOIN signature ON event.signature = signature.sig_id  LEFT JOIN sig_reference ON sig_reference.sig_id = signature.sig_id INNER JOIN sig_class ON signature.sig_class_id = sig_class.sig_class_id LEFT JOIN reference ON sig_reference.ref_id = reference.ref_id LEFT JOIN reference_system ON reference.ref_system_id = reference_system.ref_system_id INNER JOIN iphdr ON event.cid = iphdr.cid LEFT JOIN tcphdr ON event.cid = tcphdr.cid INNER JOIN sensor ON event.sid = sensor.sid WHERE event.cid > ? ORDER BY event.cid";

    /**
     * constructor
     * 
     * @param host
     *            host IP-Address
     * @param name
     *            database name
     * @param user
     *            database user name
     * @param pass
     *            database password
     */
    public SnortSqlPollingThread(Properties pr) {

        // initialize properties
        initProperties(pr);

        // create database connection
        initDatabaseConnection(mSnortDatabaseHost, mSnortDatabaseName, mSnortDatabaseUser,
                mSnortDatabasePass);

        // build prepared statement
        try {
            mPrepStatement = mConnect.prepareStatement(mRawQuery);
        } catch (SQLException e) {
            e.printStackTrace();
            IfMapClient
                    .exit("snort-sql-polling-thread encountered an error while building prepared db-statement!");
        }
    }

    @Override
    protected void initProperties(Properties props) {
        // load properties
        mSnortDatabaseHost = props.getProperty("snort.database.host", null);
        mSnortDatabaseName = props.getProperty("snort.database.name", null);
        mSnortDatabaseUser = props.getProperty("snort.database.user", null);
        mSnortDatabasePass = props.getProperty("snort.database.pass", null);

        // check properties
        if (mSnortDatabaseHost == null || mSnortDatabaseHost.length() == 0) {
            IfMapClient
                    .exit("error while initializing SnortSqlPolling Thread - property DATABASE-HOST cannot be null or empty!");
        } else if (mSnortDatabaseName == null || mSnortDatabaseName.length() == 0) {
            IfMapClient
                    .exit("error while initializing SnortSqlPolling Thread: - property DATABASE-NAME cannot be null or empty!");
        } else if (mSnortDatabaseUser == null || mSnortDatabaseUser.length() == 0) {
            IfMapClient
                    .exit("error while initializing SnortSqlPolling Thread - property DATABASE-USER cannot be null or empty!");
        } else if (mSnortDatabasePass == null || mSnortDatabasePass.length() == 0) {
            IfMapClient
                    .exit("error while initializing SnortSqlPolling Thread - property DATABASE-PASS cannot be null or empty!");
        }
    }

    /**
     * initializes the database connection
     * 
     * @param host
     *            host IP-Address
     * @param name
     *            database name
     * @param user
     *            database user name
     * @param pass
     *            database password
     */
    private void initDatabaseConnection(String host, String name, String user, String pass) {
        // initialize DB-connection
        IfMapClient.LOGGER.config("initalizing snort-database connection with host " + host
                + " and name " + name + " as user " + user + "...");
        try {
            // load the MySQL driver
            Class.forName("com.mysql.jdbc.Driver");

            // Setup the connection with the DB
            mConnect = DriverManager.getConnection("jdbc:mysql://" + host + "/" + name, user, pass);

        } catch (Exception e) {
            e.printStackTrace();
            IfMapClient
                    .exit("snort-sql-polling-thread encountered an error while connecting to snort-database "
                            + name
                            + " at host "
                            + host
                            + " with user "
                            + user
                            + "...please check your configuration");
        }
    }

    @Override
    public void run() {
        /**
         * thread-run()-method
         */
        while (running) {
            if (!pausing) {
                try {
                    IfMapClient.LOGGER.info("checking snort database for updates...");
                    mConnect.setAutoCommit(false);

                    // set current cid parameter to prepared statement
                    // logger.info("executing db-query with cId: " + lastCid);
                    mPrepStatement.setInt(1, mLastCid);
                    ResultSet resultSet = mPrepStatement.executeQuery();
                    mConnect.commit();

                    // analyze result-set and notify observers
                    notify(resultSet);

                    // sleep for a while
                    try {
                        Thread.sleep(sleepTime);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                        IfMapClient
                                .exit("snort-sql-polling-thread encountered an error while thread is trying to get some sleep...please be quit ;-)");
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    IfMapClient
                            .exit("snort-sql-polling-thread encountered an error while executing the db query");
                }
            }
        }
    }

    @Override
    public void notify(Object o) {
        if (o != null) {
            // cast passed in object back to DB-result-set
            ResultSet result = (ResultSet) o;

            // create attribute-value pairs from result-set
            ArrayList<HashMap<String, String>> resultList = getResultList(result);

            // in case of new entries, notify observer and pass the new entries
            if (resultList != null && resultList.size() > 0) {
                // hold DB-Thread
                pausing = true;

                // set internal database id (cid) of last result in list, used
                // to
                // determine query-starting point for next DB-poll-cycle lastCid
                // =
                mLastCid = new Integer(resultList.get(resultList.size() - 1).get("cid")).intValue();

                // notify the observer and pass new entries
                setChanged();
                notifyObservers(resultList);
            } else {
                IfMapClient.LOGGER.info("no new entries in snort-database, not calling observer");
            }
        } else {
            IfMapClient.LOGGER
                    .warning("retrieved snort-database result is null, not calling observer");
        }
    }

    /**
     * map result-set from database-query to a list of attribute-value-pairs
     * (realized as ArrayList of HashMaps for now)
     * 
     * @param resultSet
     *            result-set of database query
     * 
     * @return ArrayList list containing the DB-Entries as HashMaps
     */
    private ArrayList<HashMap<String, String>> getResultList(ResultSet resultSet) {
        int numColumns = 0;
        String[] columnNames = null;
        ArrayList<HashMap<String, String>> resultList = new ArrayList<HashMap<String, String>>();

        if (resultSet != null) {
            try {
                numColumns = resultSet.getMetaData().getColumnCount();

                // Get the column names; column indices start from 1
                columnNames = new String[numColumns];
                for (int i = 1; i < numColumns + 1; i++) {
                    String columnName = resultSet.getMetaData().getColumnName(i);
                    columnNames[i - 1] = columnName;
                }
            } catch (SQLException e) {
                e.printStackTrace();
            }

            // create new event-object from result-set-entries
            try {

                while (resultSet.next()) {
                    HashMap<String, String> newEntry = new HashMap<String, String>();
                    if (columnNames != null) {
                        for (int i = 0; i < columnNames.length; i++) {
                            newEntry.put(columnNames[i], resultSet.getString(columnNames[i]));
                        }
                    }
                    resultList.add(newEntry);
                }

            } catch (SQLException e) {
                e.printStackTrace();
                IfMapClient.LOGGER
                        .info("snort-database-polling-thread could not create result from read in database values!");
            }
        } else {
            IfMapClient.LOGGER
                    .info("snort-database-polling-thread could not create result from read in database values!");
        }

        return resultList;
    }
}