/* 
 * IfMapClient.java        0.1.4 12/02/16
 * 
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.main;

import de.esukom.decoit.ifmapclient.config.BasicPropertiesReader;
import de.esukom.decoit.ifmapclient.config.GeneralConfig;
import de.esukom.decoit.ifmapclient.config.GeneralPropertiesReader;
import de.esukom.decoit.ifmapclient.iptables.IPTablesFacade;
import de.esukom.decoit.ifmapclient.logging.Logging;
import de.esukom.decoit.ifmapclient.mappingfactory.MappingFactory;
import de.esukom.decoit.ifmapclient.mappingfactory.MappingResult;
import de.esukom.decoit.ifmapclient.messaging.IFMAPMessagingFacade;
import de.esukom.decoit.ifmapclient.pollingthreads.PollingThread;
import de.esukom.decoit.ifmapclient.util.Toolbox;

import java.io.File;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Observable;
import java.util.Observer;
import java.util.Properties;
import java.util.logging.Logger;
import java.util.regex.Matcher;

/**
 * Entry-Class. Here we initiate all components and stick them together. This Class is mostly based on the DHCP-IFMAP-CLIENT Implementation
 * of the FHH
 * 
 * @version 0.1.4
 * @author Tobias, FHH/TRUST
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class IfMapClient implements Observer {

    // logging
    public static final Logger LOGGER = Logging.getTheLogger();

    // ip-address of MAP-Server
    public static String sMapServerIP = null;

    // ip-address of this client
    public static String sClientIP = null;

    // command line parameters (for configuring)
    public static String[] sInputConfigParams = null;

    // thread for polling logs for new entries
    private PollingThread mPollingThread;

    // path to main configuration file
    private final String mDefaultConfigPath = "config/config.properties";

    /**
     * Constructor, initializes configuration-properties and the polling-thread-object
     * 
     * @param args
     *            command-line parameters, not used in this version
     */
    public IfMapClient(String[] args) {
        Toolbox.printLogo();

        // check command line parameters and set them if present
        if (args != null && args.length > 0) {
            sInputConfigParams = args;
        }

        init();
    }

    /**
     * initialize all required application parameters and components
     * 
     * @param args
     *            parameters from command line
     */
    private void init() {
        // initialize configuration-files
        initConfigFiles();

        // parse the ip-address of map-server from configuration-file
        Matcher ipMatcher = Toolbox.getRegExPattern(Toolbox.REGEX_GENERAL_IP4).matcher(GeneralConfig.MAPSERVER_URL);

        if (ipMatcher.find()) {
            sMapServerIP = ipMatcher.group();
        } else {
            exit("error while parsing ip-address of map-server from general configuration...please check your configuration");
        }

        // get ip-address of the machine this client runs on from
        // config.properties. its ugly this way, but currently i can
        // not find a reliable(!) way of getting it automatically from
        // a linux-based VM...
        sClientIP = GeneralConfig.APPLICATION_IPADDRESS;
        if (!sClientIP.equals("unknown")) {
            LOGGER.config("IP-Address for this Machine (as defined in config.properties) -> " + sClientIP);
        } else {
            exit("no ip-address for this machine found in config.properties...please check your configuration");
        }

        // initialize messaging-facade
        initMessagingFacade();

        // initialize log-polling threads
        initPollingThreads();

        // if ip-tables component is activated, initialize the
        // ip-tables-facade and execute ip-tables start-rules
        if (mPollingThread instanceof de.esukom.decoit.ifmapclient.pollingthreads.IPTablesULogFilePollingThread) {
            initIpTablesFacade();
            IPTablesFacade.getInstance().executeIPTableStartRules(IFMAPMessagingFacade.getInstance().mapServerIP);
        }

        // set shutdown exit-hook
        initShutdownHook();
    }

    /**
     * load and initialize configuration files
     */
    private void initConfigFiles() {
        if ((new File(mDefaultConfigPath)).exists()) {
            LOGGER.info("loading configuration files from " + mDefaultConfigPath + "...");

            // load general properties
            GeneralPropertiesReader.loadProperties(mDefaultConfigPath);

            // load polling properties
            BasicPropertiesReader.loadProperties(GeneralConfig.APPLICATION_POLLINGCONFIG_PATH);

            // load mapping properties
            BasicPropertiesReader.loadProperties(GeneralConfig.APPLICATION_MAPPINGCONFIG_PATH);

        } else {
            exit("could not find main configuration file at: " + mDefaultConfigPath + "...please check if it exists");
        }
    }

    /**
     * initialize the if-map-messaging-facade
     * 
     * @param sMapServerIP
     *            ip-address of the MAP-Server
     */
    private void initMessagingFacade() {
        LOGGER.info("initializing messaging facade with map-server-ip " + sMapServerIP + "...");
        IFMAPMessagingFacade.getInstance().init(sMapServerIP);
    }

    /**
     * initialize the ip-tables-facade
     */
    private void initIpTablesFacade() {
        LOGGER.info("initializing iptables facade...");
        BasicPropertiesReader.loadProperties("config/iptables/enforcement.properties");
        IPTablesFacade.getInstance().init(BasicPropertiesReader.getProperties());
    }

    /**
     * initialize polling threads
     */
    private void initPollingThreads() {
        LOGGER.info("initializing polling thread using class: " + GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME + "...");

        // check polling properties from polling-config file
        if (BasicPropertiesReader.getProperties() == null) {
            exit("no properties found...check your configuration in polling.properties");
        }

        // initialize Polling-Thread from configuration-file
        Class<?> pollingClass = null;
        try {
            pollingClass = Class.forName(GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME);
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
            exit("no valid polling-class found...check your configuration in config.properties");
        }

        Constructor<?> constructor = null;
        try {
            constructor = pollingClass.getConstructor(new Class[] { Properties.class });
        } catch (Exception e) {
            e.printStackTrace();
            exit("could not get constructor for class " + GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME + "...check your configuration");
        }

        try {
            mPollingThread = (PollingThread) constructor.newInstance(BasicPropertiesReader.getProperties());
        } catch (Exception e) {
            e.printStackTrace();
            exit("could not create new polling-class from " + GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME
                    + "...check your configuration");
        }

        // set thread sleep time for each cycle (in seconds!)
        mPollingThread.sleepTime = GeneralConfig.APPLICATION_POLLING_INTERVAL * 1000;

        // add corresponding observer-entity
        mPollingThread.addObserver(this);
    }

    /**
     * set the shutdown-hook for send endSession()
     */
    private void initShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            public void run() {
                try {
                    IFMAPMessagingFacade.getInstance().sendEndSessionRequest();
                } catch (Exception e) {
                    e.printStackTrace();
                    exit("error while initializing shutdown-hook!");
                }
            }
        });
    }

    /**
     * Starts the program
     */
    public void start() {
        LOGGER.info("trying to connect to MAP-Server at: " + GeneralConfig.MAPSERVER_URL + "...");

        // get number of connection-retries from configuration
        int retries = GeneralConfig.APPLICATION_CONNECTION_RETRYCOUNT;
        int retriesLeft = retries;

        // establish session with MAP-Server and send initial purgePublish
        boolean connected = false;
        while (retriesLeft > 0) {
            IFMAPMessagingFacade.getInstance().sendNewSessionRequest();
            IFMAPMessagingFacade.getInstance().sendPurgePublishRequest();
            connected = true;
            if (!connected) {
                LOGGER.warning("could not established connection...retrying (" + (retriesLeft - 1) + " left)");
                retriesLeft--;
            } else {
                break;
            }
        }

        if (connected) {
            LOGGER.info("connection  established with publisher-id: " + IFMAPMessagingFacade.getInstance().getIFMAPPublisherId());
        } else {
            exit("error while sending initial purgePublish " + retries + " times...giving up!");
        }

        // start Polling-Thread
        if (mPollingThread != null) {
            LOGGER.info("starting polling-thread at " + Toolbox.now());
            mPollingThread.running = true;
            mPollingThread.pausing = false;
            new Thread(mPollingThread).start();
        } else {
            exit("polling-thread is not initialized and therefore cannot be started");
        }

        // if esukom-mode for ip-tables is activated, the arc-polling thread that listens for new
        // alert events (that can lead to an enforcement of a client) will be started here
        if (GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME
                .equals("de.esukom.decoit.ifmapclient.pollingthreads.IPTablesULogFilePollingThread")
                && GeneralConfig.APPLICATION_ARCPOLLING_ENABLED && GeneralConfig.APPLICATION_IPTABLES_MODE.equalsIgnoreCase("esukom")) {
            IFMAPMessagingFacade.getInstance().subscribeToAlertEvents();
        }
    }

    /**
     * Is called when Polling-Thread has updated results. Pauses the Thread, maps the result with help from the Mapping-Factory and passes
     * the messages to SOAPControl-Class which finally sends them to the map-server
     * 
     * @param o
     *            the observable (polling-thread) object that called update-method
     * @param arg
     *            result from polling-object (in form of a HashMap-List)
     */
    @Override
    public synchronized void update(Observable o, Object arg) {
        LOGGER.info("polling-thread notifies observer about new result!");

        // check if passed in object exists
        if (o != null) {
            // TODO: check casting...
            ArrayList<HashMap<String, String>> tmpResultList = (ArrayList<HashMap<String, String>>) arg;
            if (tmpResultList != null) {
                MappingFactory mappingFactory = null;
                LOGGER.info("creating mapping factory for: " + GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME);
                mappingFactory = getMappingFactory(tmpResultList, GeneralConfig.APPLICATION_MAPPINGCONFIG_CLASSNAME);

                // use mapping factory to map result from poller
                if (mappingFactory != null) {
                    MappingResult[] resultList = mappingFactory.getMappingResult();

                    if (resultList != null && resultList.length > 0) {
                        // in case of poll result from ip-tables-log, just post
                        // the detected data-stream
                        if (o instanceof de.esukom.decoit.ifmapclient.pollingthreads.IPTablesULogFilePollingThread) {
                            if (GeneralConfig.APPLICATION_ARCPOLLING_ENABLED) {
                                // esukom-mode
                                if (GeneralConfig.APPLICATION_IPTABLES_MODE.equalsIgnoreCase("esukom")) {
                                    // check read-out clients for allowance
                                    IFMAPMessagingFacade.getInstance().checkEntriesForAllowance(resultList);
                                }
                                // default-mode
                                else {
                                    // add new clients to client-list
                                    if (IFMAPMessagingFacade.getInstance().pollingClientList.addNewClientsToClientList(resultList)) {

                                        // update subscriptions with new entries
                                        IFMAPMessagingFacade.getInstance().updateSubscribtions();

                                        // send new events to map-server
                                        LOGGER.info("sending data to MAP-Server at " + GeneralConfig.MAPSERVER_URL);
                                        if (GeneralConfig.APPLICATION_IPTABLES_SENDDATASTREAMDETECTEDEVENT) {
                                            IFMAPMessagingFacade.getInstance().sendPublishEventRequest(resultList);
                                        }
                                    }
                                }
                            }
                        } else {
                            // send new events to map-server
                            LOGGER.info("sending data to MAP-Server at " + GeneralConfig.MAPSERVER_URL);
                            IFMAPMessagingFacade.getInstance().sendPublishEventRequest(resultList);
                            LOGGER.info("done sending data to MAP-Server at " + GeneralConfig.MAPSERVER_URL);
                        }
                    } else {
                        LOGGER.info("mapping result is null or empty, not sending anything to server");
                    }
                } else {
                    LOGGER.warning("cannot get result from mapping factory because its not initialized");
                }
            } else {
                LOGGER.warning("result from " + GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME
                        + " is null after casting, nothing to send to server!");
            }
            // }
        } else {
            LOGGER.info("result from " + GeneralConfig.APPLICATION_POLLINGCONFIG_CLASSNAME + " is null, nothing to send to server!");
        }

        // wake Polling-Thread
        mPollingThread.pausing = false;
    }

    /**
     * get the mapping factory for converting result from polling-thread to Event-Message Object
     * 
     * @param data
     *            result from polling-object
     * 
     * @return MappingFactory mapping-factory-object containing the converted result
     */
    private MappingFactory getMappingFactory(ArrayList<HashMap<String, String>> data, String mappingConfigClassName) {
        if (data != null) {
            // initialize Mapping Factory from configuration-file
            Class<?> mappingClass = null;
            try {
                mappingClass = Class.forName(mappingConfigClassName);
            } catch (ClassNotFoundException e) {
                e.printStackTrace();
                exit("could not find MappingClass...check your config.properties for errors!");
            }

            Constructor<?> constructor = null;
            try {
                constructor = mappingClass.getConstructor(new Class[] { Properties.class, ArrayList.class });
            } catch (Exception e) {
                e.printStackTrace();
                exit("could not get Constructor of MappingClass");
            }

            MappingFactory mappingFactory = null;
            try {
                mappingFactory = (MappingFactory) constructor.newInstance(BasicPropertiesReader.getProperties(), data);
            } catch (Exception e) {
                e.printStackTrace();
                exit("could not create MappingFactory");
            }

            return mappingFactory;
        } else {
            LOGGER.warning("could not create mapping factory from empty result");
            return null;
        }
    }

    /**
     * entry-point for the application
     * 
     * @param args
     *            command-line parameters, not used in this version
     */
    public static void main(String[] args) {
        IfMapClient client = new IfMapClient(args);
        client.start();
    }

    /**
     * shut down application
     */
    public static void exit(String errorMsg) {
        if (errorMsg != null) {
            LOGGER.severe("client ended due to error with message: " + errorMsg);
            System.exit(1);
        } else {
            LOGGER.info("client ended normaly...bye bye :-)");
            System.exit(0);
        }
    }
}