/* 
 * NagiosEventMappingFactory.java        0.1.4 12/02/16
 *
 * DEVELOPED BY DECOIT GMBH WITHIN THE ESUKOM-PROJECT:
 * http://www.decoit.de/
 * http://www.esukom.de/cms/front_content.php?idcat=10&lang=1
 * 
 * DERIVED FROM  THE DHCP-IFMAP-CLIENT-IMPLEMENTATION DEVELOPED BY 
 * FHH/TRUST WITHIN THE IRON-PROJECT:
 * http://trust.inform.fh-hannover.de/joomla/
 * 
 * Licensed to the Apache Software Foundation (ASF) under one 
 * or more contributor license agreements.  See the NOTICE file 
 * distributed with this work for additional information 
 * regarding copyright ownership.  The ASF licenses this file 
 * to you under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License.  You may obtain a copy of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, 
 * software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY 
 * KIND, either express or implied.  See the License for the 
 * specific language governing permissions and limitations 
 * under the License. 
 */

package de.esukom.decoit.ifmapclient.mappingfactory;

import de.esukom.decoit.ifmapclient.main.IfMapClient;
import de.esukom.decoit.ifmapclient.messaging.SOAPMessageSender;
import de.esukom.decoit.ifmapclient.util.Toolbox;

import de.fhhannover.inform.trust.ifmapj.metadata.EventType;
import de.fhhannover.inform.trust.ifmapj.metadata.Significance;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;

/**
 * concrete implementation of abstract mapping-factory for mapping values from
 * polling-threads to event objects that can be send to MAP-Server
 * 
 * @version 0.1.4
 * @author Dennis Dunekacke, Decoit GmbH
 */
public class NagiosEventMappingFactory extends MappingFactory {

    // properties from mapping config file
    private EventType mEventtypeDefault;
    private String mMagnitudeDefault;
    private String mConfidenceDefault;
    private Significance mSignificanceDefault;

    // name of keys for result-hashmap-list
    private final String mTimestampKey = "timestamp";
    private final String mAddressKey = "address";
    private final String mEventsourceKey = "source";
    private final String mEventstateKey = "state";

    public NagiosEventMappingFactory(Properties props, ArrayList<HashMap<String, String>> data) {
        super(props, data);
    }

    @Override
    protected void createMappingResult(Properties props, ArrayList<HashMap<String, String>> data) {

        // initalize mapping properties
        initProperties(props);

        // create EventMapping-Objects from incoming data
        boolean successfull = true;
        EventMappingResult event = new EventMappingResult();
        for (int i = 0; i < data.size(); i++) {
            // discovered time
            if (data.get(i).get(mTimestampKey) != null) {
                event.setDiscoveredTime(rearrangeDate(data.get(i).get(mTimestampKey)));
            } else {
                IfMapClient.LOGGER
                        .warning("mapping of nagios-event failed - cannot find value for event.discovered_time");
                successfull = false;
                break;
            }

            // ip-address
            if (data.get(i).get(mAddressKey) != null) {
                event.setIp(data.get(i).get(mAddressKey));
            } else {
                IfMapClient.LOGGER
                        .warning("mapping of nagios-event failed - cannot find value for event.ip");
                successfull = false;
                break;
            }

            // event-name
            if (data.get(i).get(mEventsourceKey) != null && data.get(i).get(mEventstateKey) != null) {
                event.setName(getEventnameValue(data.get(i).get(mEventsourceKey),
                        data.get(i).get(mEventstateKey)));
            } else {
                IfMapClient.LOGGER
                        .warning("mapping of nagios-event failed - cannot find values for event.name");
                successfull = false;
                break;
            }

            // don't really know how to map these attributes...
            if (successfull) {
                event.setIpType("IPv4");
                event.setDiscovererId(SOAPMessageSender.getInstance().getIfMapPublisherId());
                event.setSignificance(mSignificanceDefault);
                event.setEventMessageType(mEventtypeDefault);
                event.setConfidence(mConfidenceDefault);
                event.setMagnitude(mMagnitudeDefault);
            }
        }

        if (successfull) {
            // add to result-list
            super.mapResult.add(event);
            IfMapClient.LOGGER.fine("mapped nagios event has been added to result list: "
                    + event.showOnConsole());
        } else {
            // if parentsClass ("the factory") returns null, the "MainClass"
            // can catch this error. The mapResult-List of the parent class
            // has to be explicitly set to null because its already initialized
            // at this point!
            super.mapResult = null;
        }
    }

    /**
     * initialize config from property-file
     * 
     * @param conf
     *            properties-object
     */
    public void initProperties(Properties props) {
        boolean criticalErrorOcured = false;

        // event to be processed/mapped
        mEventtypeDefault = EventType.behavioralChange; // default!

        String eventTypeFromConfig = Toolbox.getStringProperty("nagios.eventmapping.eventtype",
                props, false);
        if (mEventtypeDefault == null) {
            criticalErrorOcured = true;
        }

        // map event-type-string to ifmaplibj EventType-type
        if (eventTypeFromConfig == "botnet-infection") {
            mEventtypeDefault = EventType.botnetInfection;

        } else if (eventTypeFromConfig == "cve") {
            mEventtypeDefault = EventType.cve;

        } else if (eventTypeFromConfig == "excessive-flows") {
            mEventtypeDefault = EventType.excessiveFlows;

        } else if (eventTypeFromConfig == "other") {
            mEventtypeDefault = EventType.other;

        } else if (eventTypeFromConfig == "p2p") {
            mEventtypeDefault = EventType.p2p;

        } else if (eventTypeFromConfig == "policy-violation") {
            mEventtypeDefault = EventType.policyViolation;

        } else if (eventTypeFromConfig == "worm-infection") {
            mEventtypeDefault = EventType.wormInfection;

        } else {
            IfMapClient.LOGGER
                    .config("could not load default event-type value from nagios mapping config...using default (behavioral-change)");
        }

        // magnitude mapping
        mMagnitudeDefault = Toolbox
                .getStringProperty("nagios.eventmapping.magnitude", props, false);
        if (mMagnitudeDefault == null) {
            criticalErrorOcured = true;
        }

        // confidence mapping
        mConfidenceDefault = Toolbox.getStringProperty("nagios.eventmapping.confidence", props,
                false);
        if (mConfidenceDefault == null) {
            criticalErrorOcured = true;
        }

        // significance mapping
        mSignificanceDefault = Significance.informational; // default!
        String significanceFromConfig = Toolbox.getStringProperty(
                "nagios.eventmapping.significance", props, false);

        // map significance-string to ifmaplibj significance datatype
        if (significanceFromConfig == "important") {
            mSignificanceDefault = Significance.important;

        } else if (significanceFromConfig == "critical") {
            mSignificanceDefault = Significance.critical;

        } else {
            IfMapClient.LOGGER
                    .config("could not load default significance value from nagios mapping config...using default (informational)");
        }

        if (criticalErrorOcured) {
            IfMapClient.exit("error while initializing NagiosEventMappingFactory");
        }
    }

    /**
     * get if-map conform time-stamp from passed in currentDate-String
     * 
     * @param String
     *            currentDate (YYYY/MM/DD-hh:mm:ss.S)
     * 
     * @return time-stamp in IF-MAP format
     */
    private String rearrangeDate(String currentDate) {
        // YYYY/MM/DD-hh:mm:ss.S => [0]Date [1]Time
        String[] timestamp = currentDate.split("-");

        // YYYY/MM/DD => [0]Year [1] Month [2] Day
        String[] date = timestamp[0].split("/");

        // build new timestamp-string
        return date[0] + "-" + date[1] + "-" + date[2] + "T" + timestamp[1] + "Z";
    }

    /**
     * don't know how to map this attribute... for now just use default value
     * from mapping-config
     */
    public String getEventnameValue(String name, String state) {
        if (name.startsWith("service")) {
            return "Detected Service State: " + state;
        } else if (name.startsWith("host")) {
            return "Detected Host State: " + state;
        } else {
            return "Undefined Event Name";
        }
    }
}